import { prisma } from '@/lib/db';
import { MediatorStatus } from '@prisma/client';
import { NotFoundError, ForbiddenError, ValidationError } from '@/modules/shared/errors';
import { createAuditLog } from '@/modules/shared/audit';

export interface UpdateProfileData {
  firstName?: string;
  lastName?: string;
  displayName?: string;
  bio?: string;
  photoUrl?: string;
  phone?: string;
  officeAddress?: string;
  certifications?: string[];
  specializations?: string[];
  yearsExperience?: number;
  languages?: string[];
}

export class MediatorProfileService {
  /**
   * Get mediator profile by ID
   */
  async getProfile(mediatorId: string) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: {
        user: {
          select: {
            email: true,
            createdAt: true,
          },
        },
        pricing: true,
      },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator profile not found');
    }

    return mediator;
  }

  /**
   * Get mediator profile by slug (for public pages)
   */
  async getProfileBySlug(slug: string) {
    const mediator = await prisma.mediator.findUnique({
      where: { slug },
      include: {
        pricing: true,
      },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator profile not found');
    }

    return mediator;
  }

  /**
   * Update mediator profile
   */
  async updateProfile(
    mediatorId: string,
    userId: string,
    data: UpdateProfileData
  ) {
    // Verify mediator exists and user owns it
    const existing = await prisma.mediator.findUnique({
      where: { id: mediatorId },
    });

    if (!existing) {
      throw new NotFoundError('Mediator profile not found');
    }

    if (existing.userId !== userId) {
      throw new ForbiddenError('You do not have permission to edit this profile');
    }

    // Update profile
    const mediator = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        ...data,
        updatedAt: new Date(),
      },
    });

    // Create audit log
    await createAuditLog({
      userId,
      action: 'PROFILE_UPDATE',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { updatedFields: Object.keys(data) },
    });

    return mediator;
  }

  /**
   * Update photo URL
   */
  async updatePhoto(mediatorId: string, userId: string, photoUrl: string) {
    return this.updateProfile(mediatorId, userId, { photoUrl });
  }

  /**
   * Check if profile is complete enough to submit for approval
   */
  async isProfileComplete(mediatorId: string): Promise<boolean> {
    const mediator = await this.getProfile(mediatorId);

    // Required fields for approval
    const requiredFields = [
      mediator.bio,
      mediator.phone,
      mediator.specializations.length > 0,
      mediator.yearsExperience !== null,
    ];

    return requiredFields.every(Boolean);
  }

  /**
   * Submit profile for approval
   */
  async submitForApproval(mediatorId: string, userId: string) {
    const mediator = await this.getProfile(mediatorId);

    if (mediator.userId !== userId) {
      throw new ForbiddenError('You do not have permission to submit this profile');
    }

    // Check if already submitted or approved
    if (mediator.status !== 'PENDING_APPROVAL' && mediator.status !== 'REJECTED') {
      // Allow resubmission only if previously PENDING or REJECTED
      if (mediator.status === 'APPROVED' || mediator.status === 'PUBLISHED') {
        throw new ValidationError('Profile is already approved');
      }
    }

    // Check if profile is complete
    const isComplete = await this.isProfileComplete(mediatorId);
    if (!isComplete) {
      throw new ValidationError(
        'Profile is incomplete. Please fill in bio, phone, specializations, and years of experience.'
      );
    }

    // Update status to pending
    const updated = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'PENDING_APPROVAL',
      },
    });

    await createAuditLog({
      userId,
      action: 'PROFILE_UPDATE',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { action: 'submitted_for_approval' },
    });

    return updated;
  }

  /**
   * Get all mediators with optional status filter (admin only)
   */
  async getAllMediators(status?: MediatorStatus) {
    return prisma.mediator.findMany({
      where: status ? { status } : undefined,
      include: {
        user: {
          select: {
            email: true,
            createdAt: true,
          },
        },
        pricing: true,
        _count: {
          select: {
            cases: true,
          },
        },
      },
      orderBy: {
        createdAt: 'desc',
      },
    });
  }

  /**
   * Get pending approvals (admin only)
   */
  async getPendingApprovals() {
    return this.getAllMediators('PENDING_APPROVAL');
  }

  /**
   * Approve mediator profile (admin only)
   */
  async approveMediator(
    mediatorId: string,
    adminId: string,
    notes?: string
  ) {
    const mediator = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'APPROVED',
        approvalNotes: notes,
        approvedBy: adminId,
        approvedAt: new Date(),
      },
      include: {
        user: true,
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'MEDIATOR_APPROVE',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { notes },
    });

    return mediator;
  }

  /**
   * Reject mediator profile (admin only)
   */
  async rejectMediator(
    mediatorId: string,
    adminId: string,
    reason: string
  ) {
    if (!reason || reason.trim().length === 0) {
      throw new ValidationError('Rejection reason is required');
    }

    const mediator = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'REJECTED',
        approvalNotes: reason,
        approvedBy: adminId,
      },
      include: {
        user: true,
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'MEDIATOR_REJECT',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { reason },
    });

    return mediator;
  }

  /**
   * Publish mediator profile (admin only)
   */
  async publishMediator(mediatorId: string, adminId: string) {
    const mediator = await this.getProfile(mediatorId);

    if (mediator.status !== 'APPROVED') {
      throw new ValidationError('Only approved mediators can be published');
    }

    const updated = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'PUBLISHED',
        publishedAt: new Date(),
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'PROFILE_PUBLISH',
      resource: 'mediator',
      resourceId: mediatorId,
    });

    return updated;
  }

  /**
   * Suspend mediator profile (admin only)
   */
  async suspendMediator(
    mediatorId: string,
    adminId: string,
    reason: string
  ) {
    if (!reason || reason.trim().length === 0) {
      throw new ValidationError('Suspension reason is required');
    }

    const mediator = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'SUSPENDED',
        approvalNotes: reason,
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'MEDIATOR_SUSPEND',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { reason },
    });

    return mediator;
  }

  /**
   * Unpublish mediator (revert to approved)
   */
  async unpublishMediator(mediatorId: string, adminId: string) {
    const mediator = await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        status: 'APPROVED',
      },
    });

    await createAuditLog({
      userId: adminId,
      action: 'PROFILE_UPDATE',
      resource: 'mediator',
      resourceId: mediatorId,
      metadata: { action: 'unpublished' },
    });

    return mediator;
  }

  /**
   * Get published mediators (for public directory)
   */
  async getPublishedMediators(filters?: {
    specialization?: string;
    minExperience?: number;
    search?: string;
  }) {
    return prisma.mediator.findMany({
      where: {
        status: 'PUBLISHED',
        ...(filters?.specialization && {
          specializations: {
            has: filters.specialization,
          },
        }),
        ...(filters?.minExperience && {
          yearsExperience: {
            gte: filters.minExperience,
          },
        }),
        ...(filters?.search && {
          OR: [
            {
              firstName: {
                contains: filters.search,
                mode: 'insensitive',
              },
            },
            {
              lastName: {
                contains: filters.search,
                mode: 'insensitive',
              },
            },
            {
              bio: {
                contains: filters.search,
                mode: 'insensitive',
              },
            },
          ],
        }),
      },
      select: {
        id: true,
        slug: true,
        firstName: true,
        lastName: true,
        displayName: true,
        photoUrl: true,
        bio: true,
        specializations: true,
        yearsExperience: true,
        certifications: true,
        languages: true,
        isGstRegistered: true,
        pricing: {
          select: {
            hourlyRate: true,
            hourlyRateBasis: true,
            preMediationFee: true,
          },
        },
      },
      orderBy: {
        publishedAt: 'desc',
      },
    });
  }
}
