'use server';

import { MediatorSearchService } from './service';

const searchService = new MediatorSearchService();

interface ActionResult {
  error?: string;
  success?: boolean;
  data?: any;
}

/**
 * Search mediators with filters
 */
export async function searchMediatorsAction(filters: {
  specialization?: string;
  onlineAvailable?: boolean;
  minPrice?: number;
  maxPrice?: number;
  maxBudget?: number;
  languages?: string[];
  minExperience?: number;
  search?: string;
}): Promise<ActionResult> {
  try {
    const mediators = await searchService.searchMediators(filters);
    return { success: true, data: mediators };
  } catch (error) {
    return { error: 'Failed to search mediators' };
  }
}

/**
 * Calculate price estimate for a mediator
 */
export async function calculatePriceEstimateAction(
  mediatorId: string,
  estimate: {
    estimatedHours: number;
    numberOfParties: number;
    requiresTravel?: boolean;
    travelDistance?: number;
    requiresAccommodation?: boolean;
  }
): Promise<ActionResult> {
  try {
    const priceEstimate = await searchService.calculatePriceEstimate(
      mediatorId,
      estimate
    );
    return { success: true, data: priceEstimate };
  } catch (error) {
    return { 
      error: error instanceof Error ? error.message : 'Failed to calculate estimate' 
    };
  }
}

/**
 * Get suggested mediators based on requirements
 */
export async function getSuggestedMediatorsAction(
  filters: {
    specialization?: string;
    maxBudget?: number;
    languages?: string[];
    minExperience?: number;
  },
  estimate: {
    estimatedHours: number;
    numberOfParties: number;
    requiresTravel?: boolean;
    travelDistance?: number;
    requiresAccommodation?: boolean;
  }
): Promise<ActionResult> {
  try {
    const suggestions = await searchService.getSuggestedMediators(
      filters,
      estimate
    );
    return { success: true, data: suggestions };
  } catch (error) {
    return { error: 'Failed to get suggestions' };
  }
}

/**
 * Get price ranges for filters
 */
export async function getPriceRangesAction(): Promise<ActionResult> {
  try {
    const ranges = await searchService.getPriceRanges();
    return { success: true, data: ranges };
  } catch (error) {
    return { error: 'Failed to get price ranges' };
  }
}

/**
 * Get available specializations
 */
export async function getAvailableSpecializationsAction(): Promise<ActionResult> {
  try {
    const specializations = await searchService.getAvailableSpecializations();
    return { success: true, data: specializations };
  } catch (error) {
    return { error: 'Failed to get specializations' };
  }
}

/**
 * Get available languages
 */
export async function getAvailableLanguagesAction(): Promise<ActionResult> {
  try {
    const languages = await searchService.getAvailableLanguages();
    return { success: true, data: languages };
  } catch (error) {
    return { error: 'Failed to get languages' };
  }
}

/**
 * Compare multiple mediators
 */
export async function compareMediatorsAction(
  mediatorIds: string[],
  estimate: {
    estimatedHours: number;
    numberOfParties: number;
    requiresTravel?: boolean;
    travelDistance?: number;
    requiresAccommodation?: boolean;
  }
): Promise<ActionResult> {
  try {
    const comparisons = await searchService.compareMediators(
      mediatorIds,
      estimate
    );
    return { success: true, data: comparisons };
  } catch (error) {
    return { error: 'Failed to compare mediators' };
  }
}
