import { prisma } from '@/lib/db';
import { Prisma } from '@prisma/client';

export interface SearchFilters {
  specialization?: string;
  onlineAvailable?: boolean;
  minPrice?: number;
  maxPrice?: number;
  maxBudget?: number;
  location?: string;
  languages?: string[];
  minExperience?: number;
  search?: string;
}

export interface CaseEstimate {
  estimatedHours: number;
  numberOfParties: number;
  requiresTravel?: boolean;
  travelDistance?: number;
  requiresAccommodation?: boolean;
}

export interface PriceEstimate {
  mediatorId: string;
  mediatorName: string;
  mediatorSlug: string;
  
  // Fee breakdown
  preMediationFee: number;
  hourlyCharges: number;
  travelCost: number;
  accommodationCost: number;
  subtotal: number;
  
  // GST
  gstAmount: number;
  totalIncGst: number;
  isGstRegistered: boolean;
  
  // Pricing details
  hourlyRate: number;
  hourlyRateBasis: 'PER_PARTY' | 'TOTAL';
  estimatedHours: number;
  numberOfParties: number;
  
  // Scoring for recommendations
  score?: number;
}

export class MediatorSearchService {
  /**
   * Search mediators with filters
   */
  async searchMediators(filters: SearchFilters) {
    const where: Prisma.MediatorWhereInput = {
      status: 'PUBLISHED', // Only published mediators
      
      // Specialization filter
      ...(filters.specialization && {
        specializations: {
          has: filters.specialization,
        },
      }),
      
      // Language filter
      ...(filters.languages && filters.languages.length > 0 && {
        languages: {
          hasSome: filters.languages,
        },
      }),
      
      // Experience filter
      ...(filters.minExperience && {
        yearsExperience: {
          gte: filters.minExperience,
        },
      }),
      
      // Search by name or bio
      ...(filters.search && {
        OR: [
          {
            firstName: {
              contains: filters.search,
              mode: 'insensitive',
            },
          },
          {
            lastName: {
              contains: filters.search,
              mode: 'insensitive',
            },
          },
          {
            displayName: {
              contains: filters.search,
              mode: 'insensitive',
            },
          },
          {
            bio: {
              contains: filters.search,
              mode: 'insensitive',
            },
          },
        ],
      }),
    };

    const mediators = await prisma.mediator.findMany({
      where,
      include: {
        pricing: true,
      },
      orderBy: [
        { publishedAt: 'desc' },
        { yearsExperience: 'desc' },
      ],
    });

    // Apply price filters if provided
    let filtered = mediators;

    if (filters.minPrice !== undefined || filters.maxPrice !== undefined) {
      filtered = filtered.filter(mediator => {
        if (!mediator.pricing) return false;
        
        const hourlyRate = Number(mediator.pricing.hourlyRate);
        
        if (filters.minPrice !== undefined && hourlyRate < filters.minPrice) {
          return false;
        }
        
        if (filters.maxPrice !== undefined && hourlyRate > filters.maxPrice) {
          return false;
        }
        
        return true;
      });
    }

    return filtered;
  }

  /**
   * Calculate price estimate for a mediator based on case requirements
   */
  async calculatePriceEstimate(
    mediatorId: string,
    estimate: CaseEstimate
  ): Promise<PriceEstimate> {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: {
        pricing: true,
      },
    });

    if (!mediator) {
      throw new Error('Mediator not found');
    }

    if (!mediator.pricing) {
      throw new Error('Mediator has not configured pricing');
    }

    const pricing = mediator.pricing;
    const hourlyRate = Number(pricing.hourlyRate);
    
    // Pre-mediation fee
    const preMediationFee = Number(pricing.preMediationFee || 0);

    // Hourly charges
    let hourlyCharges = 0;
    if (pricing.hourlyRateBasis === 'PER_PARTY') {
      hourlyCharges = hourlyRate * estimate.estimatedHours * estimate.numberOfParties;
    } else {
      hourlyCharges = hourlyRate * estimate.estimatedHours;
    }

    // Travel costs
    let travelCost = 0;
    if (estimate.requiresTravel && estimate.travelDistance) {
      const includedKm = pricing.travelIncludedKm || 0;
      const extraKm = Math.max(0, estimate.travelDistance - includedKm);
      const ratePerKm = Number(pricing.travelRatePerKm || 0);
      travelCost = extraKm * ratePerKm;
    }

    // Accommodation
    let accommodationCost = 0;
    if (estimate.requiresAccommodation && pricing.accommodationRate) {
      accommodationCost = Number(pricing.accommodationRate);
    }

    // Subtotal
    const subtotal = preMediationFee + hourlyCharges + travelCost + accommodationCost;

    // GST calculation
    let gstAmount = 0;
    let totalIncGst = subtotal;

    if (mediator.isGstRegistered) {
      gstAmount = subtotal * 0.10; // 10% GST
      totalIncGst = subtotal + gstAmount;
    }

    return {
      mediatorId: mediator.id,
      mediatorName: mediator.displayName || `${mediator.firstName} ${mediator.lastName}`,
      mediatorSlug: mediator.slug,
      
      preMediationFee,
      hourlyCharges,
      travelCost,
      accommodationCost,
      subtotal,
      
      gstAmount,
      totalIncGst,
      isGstRegistered: mediator.isGstRegistered,
      
      hourlyRate,
      hourlyRateBasis: pricing.hourlyRateBasis,
      estimatedHours: estimate.estimatedHours,
      numberOfParties: estimate.numberOfParties,
    };
  }

  /**
   * Get suggested mediators based on budget and requirements
   */
  async getSuggestedMediators(
    filters: SearchFilters,
    estimate: CaseEstimate
  ) {
    // Get all matching mediators
    const mediators = await this.searchMediators(filters);

    // Calculate estimates for each
    const estimates = await Promise.all(
      mediators.map(async (mediator) => {
        const priceEstimate = await this.calculatePriceEstimate(
          mediator.id,
          estimate
        );

        // Calculate score based on multiple factors
        let score = 0;

        // Experience score (0-30 points)
        const experience = mediator.yearsExperience || 0;
        score += Math.min(experience * 2, 30);

        // Specialization match score (0-20 points)
        if (filters.specialization && mediator.specializations.includes(filters.specialization)) {
          score += 20;
        }

        // Budget fit score (0-30 points)
        if (filters.maxBudget) {
          const budgetFit = filters.maxBudget - priceEstimate.totalIncGst;
          if (budgetFit >= 0) {
            // Within budget - full points
            score += 30;
          } else if (budgetFit > -500) {
            // Slightly over budget - partial points
            score += 15;
          }
        } else {
          score += 15; // Neutral score if no budget specified
        }

        // Price competitiveness score (0-20 points)
        // Lower price = higher score
        const priceScore = Math.max(0, 20 - (priceEstimate.totalIncGst / 100));
        score += Math.min(priceScore, 20);

        return {
          mediator,
          estimate: priceEstimate,
          score,
        };
      })
    );

    // Filter by budget if specified
    let filtered = estimates;
    if (filters.maxBudget) {
      filtered = estimates.filter(
        item => item.estimate.totalIncGst <= filters.maxBudget
      );
    }

    // Sort by score (highest first)
    filtered.sort((a, b) => b.score - a.score);

    return filtered;
  }

  /**
   * Get price ranges for all published mediators
   */
  async getPriceRanges() {
    const mediators = await prisma.mediator.findMany({
      where: {
        status: 'PUBLISHED',
        pricing: {
          isNot: null,
        },
      },
      include: {
        pricing: true,
      },
    });

    if (mediators.length === 0) {
      return {
        minHourlyRate: 0,
        maxHourlyRate: 0,
        avgHourlyRate: 0,
      };
    }

    const rates = mediators
      .map(m => Number(m.pricing!.hourlyRate))
      .filter(r => r > 0);

    return {
      minHourlyRate: Math.min(...rates),
      maxHourlyRate: Math.max(...rates),
      avgHourlyRate: rates.reduce((sum, r) => sum + r, 0) / rates.length,
    };
  }

  /**
   * Get all unique specializations from published mediators
   */
  async getAvailableSpecializations() {
    const mediators = await prisma.mediator.findMany({
      where: {
        status: 'PUBLISHED',
      },
      select: {
        specializations: true,
      },
    });

    const allSpecializations = mediators.flatMap(m => m.specializations);
    const unique = [...new Set(allSpecializations)];
    
    return unique.sort();
  }

  /**
   * Get all unique languages from published mediators
   */
  async getAvailableLanguages() {
    const mediators = await prisma.mediator.findMany({
      where: {
        status: 'PUBLISHED',
      },
      select: {
        languages: true,
      },
    });

    const allLanguages = mediators.flatMap(m => m.languages);
    const unique = [...new Set(allLanguages)];
    
    return unique.sort();
  }

  /**
   * Get mediator details with pricing for display
   */
  async getMediatorWithPricing(mediatorId: string) {
    return prisma.mediator.findUnique({
      where: {
        id: mediatorId,
        status: 'PUBLISHED',
      },
      include: {
        pricing: true,
        _count: {
          select: {
            cases: true,
          },
        },
      },
    });
  }

  /**
   * Compare multiple mediators
   */
  async compareMediators(
    mediatorIds: string[],
    estimate: CaseEstimate
  ) {
    const mediators = await prisma.mediator.findMany({
      where: {
        id: {
          in: mediatorIds,
        },
        status: 'PUBLISHED',
      },
      include: {
        pricing: true,
      },
    });

    const comparisons = await Promise.all(
      mediators.map(async (mediator) => {
        const priceEstimate = await this.calculatePriceEstimate(
          mediator.id,
          estimate
        );

        return {
          mediator,
          estimate: priceEstimate,
        };
      })
    );

    return comparisons;
  }
}
