'use server';

import { redirect } from 'next/navigation';
import { revalidatePath } from 'next/cache';
import { getCurrentUser } from '@/modules/auth/actions';
import { StripeConnectService } from '../connect/service';
import { PaymentProcessingService } from '../processing/service';

const connectService = new StripeConnectService();
const paymentService = new PaymentProcessingService();

interface ActionResult {
  error?: string;
  success?: boolean;
  url?: string;
  data?: any;
}

/**
 * Start Stripe Connect onboarding
 */
export async function startConnectOnboardingAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    let accountId = user.mediator.stripeAccountId;

    // Create account if doesn't exist
    if (!accountId) {
      const account = await connectService.createConnectAccount(
        user.mediator.id,
        user.id
      );
      accountId = account.id;
    }

    // Create onboarding link
    const accountLink = await connectService.createAccountLink(
      user.mediator.id,
      accountId
    );

    return { success: true, url: accountLink.url };
  } catch (error) {
    console.error('Connect onboarding error:', error);
    return { error: error instanceof Error ? error.message : 'Failed to start onboarding' };
  }
}

/**
 * Check Connect onboarding status
 */
export async function checkConnectStatusAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    if (!user.mediator.stripeAccountId) {
      return { success: false, data: { onboarded: false } };
    }

    const isOnboarded = await connectService.updateOnboardingStatus(
      user.mediator.id,
      user.id
    );

    revalidatePath('/mediator/payments');

    return {
      success: true,
      data: { onboarded: isOnboarded },
    };
  } catch (error) {
    console.error('Status check error:', error);
    return { error: 'Failed to check status' };
  }
}

/**
 * Get Connect account status
 */
export async function getConnectAccountStatusAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    if (!user.mediator.stripeAccountId) {
      return { success: true, data: null };
    }

    const status = await connectService.getAccountStatus(
      user.mediator.stripeAccountId
    );

    return { success: true, data: status };
  } catch (error) {
    console.error('Account status error:', error);
    return { error: 'Failed to get account status' };
  }
}

/**
 * Create Stripe dashboard login link
 */
export async function createDashboardLoginLinkAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    if (!user.mediator.stripeAccountId) {
      return { error: 'No Stripe account found' };
    }

    const loginLink = await connectService.createLoginLink(
      user.mediator.stripeAccountId
    );

    return { success: true, url: loginLink.url };
  } catch (error) {
    console.error('Login link error:', error);
    return { error: 'Failed to create login link' };
  }
}

/**
 * Create deposit payment intent
 */
export async function createDepositPaymentAction(caseId: string): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || !user.client) {
      return { error: 'Unauthorized' };
    }

    const result = await paymentService.createDepositPayment(
      caseId,
      user.client.id,
      user.id
    );

    revalidatePath(`/client/cases/${caseId}`);

    return {
      success: true,
      data: {
        paymentIntentId: result.paymentIntent.id,
        clientSecret: result.paymentIntent.client_secret,
        amount: result.paymentIntent.amount,
      },
    };
  } catch (error) {
    console.error('Deposit payment error:', error);
    return { error: error instanceof Error ? error.message : 'Failed to create payment' };
  }
}

/**
 * Create balance payment intent
 */
export async function createBalancePaymentAction(
  caseId: string,
  amount: number,
  description?: string
): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || !user.client) {
      return { error: 'Unauthorized' };
    }

    const result = await paymentService.createBalancePayment(
      caseId,
      user.client.id,
      amount,
      user.id,
      description
    );

    revalidatePath(`/client/cases/${caseId}`);

    return {
      success: true,
      data: {
        paymentIntentId: result.paymentIntent.id,
        clientSecret: result.paymentIntent.client_secret,
        amount: result.paymentIntent.amount,
      },
    };
  } catch (error) {
    console.error('Balance payment error:', error);
    return { error: error instanceof Error ? error.message : 'Failed to create payment' };
  }
}

/**
 * Get mediator payment stats
 */
export async function getMediatorPaymentStatsAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const stats = await paymentService.getMediatorPaymentStats(user.mediator.id);

    return { success: true, data: stats };
  } catch (error) {
    console.error('Payment stats error:', error);
    return { error: 'Failed to get payment stats' };
  }
}

/**
 * Get mediator payments
 */
export async function getMediatorPaymentsAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'MEDIATOR' || !user.mediator) {
      return { error: 'Unauthorized' };
    }

    const payments = await paymentService.getMediatorPayments(user.mediator.id);

    return { success: true, data: payments };
  } catch (error) {
    console.error('Get payments error:', error);
    return { error: 'Failed to get payments' };
  }
}

/**
 * Get platform payment stats (admin)
 */
export async function getPlatformPaymentStatsAction(): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    const stats = await paymentService.getPlatformStats();

    return { success: true, data: stats };
  } catch (error) {
    console.error('Platform stats error:', error);
    return { error: 'Failed to get platform stats' };
  }
}

/**
 * Get all payments (admin)
 */
export async function getAllPaymentsAction(filters?: {
  status?: string;
  type?: string;
  mediatorId?: string;
}): Promise<ActionResult> {
  try {
    const user = await getCurrentUser();
    if (!user || user.role !== 'ADMIN') {
      return { error: 'Unauthorized' };
    }

    const payments = await paymentService.getAllPayments(filters);

    return { success: true, data: payments };
  } catch (error) {
    console.error('Get all payments error:', error);
    return { error: 'Failed to get payments' };
  }
}
