import { stripe, STRIPE_CONFIG } from '@/lib/stripe';
import { prisma } from '@/lib/db';
import { createAuditLog } from '@/modules/shared/audit';
import { NotFoundError, ValidationError } from '@/modules/shared/errors';

export class StripeConnectService {
  /**
   * Create a Stripe Connect Express account for a mediator
   */
  async createConnectAccount(mediatorId: string, userId: string) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
      include: { user: true },
    });

    if (!mediator) {
      throw new NotFoundError('Mediator not found');
    }

    // Check if account already exists
    if (mediator.stripeAccountId) {
      throw new ValidationError('Stripe Connect account already exists');
    }

    // Create Express account
    const account = await stripe.accounts.create({
      type: STRIPE_CONFIG.CONNECT_ACCOUNT_TYPE,
      country: 'AU',
      email: mediator.user.email,
      capabilities: {
        card_payments: { requested: true },
        transfers: { requested: true },
      },
      business_type: 'individual',
      metadata: {
        mediatorId: mediator.id,
        userId: userId,
      },
    });

    // Update mediator with account ID
    await prisma.mediator.update({
      where: { id: mediatorId },
      data: {
        stripeAccountId: account.id,
      },
    });

    await createAuditLog({
      userId,
      action: 'PROFILE_UPDATE',
      resource: 'stripe_connect',
      resourceId: account.id,
      metadata: { mediatorId, action: 'account_created' },
    });

    return account;
  }

  /**
   * Create an account link for onboarding
   */
  async createAccountLink(mediatorId: string, accountId: string) {
    const accountLink = await stripe.accountLinks.create({
      account: accountId,
      refresh_url: STRIPE_CONFIG.getConnectRefreshUrl(mediatorId),
      return_url: STRIPE_CONFIG.getConnectReturnUrl(mediatorId),
      type: 'account_onboarding',
    });

    return accountLink;
  }

  /**
   * Get account status
   */
  async getAccountStatus(accountId: string) {
    const account = await stripe.accounts.retrieve(accountId);
    
    return {
      id: account.id,
      chargesEnabled: account.charges_enabled,
      payoutsEnabled: account.payouts_enabled,
      detailsSubmitted: account.details_submitted,
      requirements: account.requirements,
    };
  }

  /**
   * Check if account is fully onboarded
   */
  async isAccountOnboarded(accountId: string): Promise<boolean> {
    const account = await stripe.accounts.retrieve(accountId);
    return account.charges_enabled && account.details_submitted;
  }

  /**
   * Update mediator onboarding status
   */
  async updateOnboardingStatus(mediatorId: string, userId: string) {
    const mediator = await prisma.mediator.findUnique({
      where: { id: mediatorId },
    });

    if (!mediator?.stripeAccountId) {
      throw new NotFoundError('No Stripe account found');
    }

    const isOnboarded = await this.isAccountOnboarded(mediator.stripeAccountId);

    if (isOnboarded && !mediator.stripeOnboarded) {
      await prisma.mediator.update({
        where: { id: mediatorId },
        data: {
          stripeOnboarded: true,
          stripeOnboardedAt: new Date(),
        },
      });

      await createAuditLog({
        userId,
        action: 'PROFILE_UPDATE',
        resource: 'stripe_connect',
        resourceId: mediator.stripeAccountId,
        metadata: { mediatorId, action: 'onboarding_completed' },
      });
    }

    return isOnboarded;
  }

  /**
   * Create login link for existing account (to update details)
   */
  async createLoginLink(accountId: string) {
    const loginLink = await stripe.accounts.createLoginLink(accountId);
    return loginLink;
  }

  /**
   * Get all payouts for a mediator
   */
  async getPayouts(accountId: string, limit: number = 10) {
    const payouts = await stripe.payouts.list(
      { limit },
      { stripeAccount: accountId }
    );
    return payouts.data;
  }

  /**
   * Get balance for a connect account
   */
  async getBalance(accountId: string) {
    const balance = await stripe.balance.retrieve({
      stripeAccount: accountId,
    });
    return balance;
  }
}
