import { stripe, STRIPE_CONFIG } from '@/lib/stripe';
import { prisma } from '@/lib/db';
import { createAuditLog } from '@/modules/shared/audit';
import { NotFoundError, ValidationError } from '@/modules/shared/errors';
import { Decimal } from '@prisma/client/runtime/library';

export interface FeeCalculation {
  grossAmount: number; // Amount client pays
  stripeFeeAmount: number; // Stripe's fee
  stripeNetAmount: number; // After Stripe fee
  platformFeeAmount: number; // Platform's fee (% of net)
  platformFeeGst: number; // GST on platform fee
  mediatorAmount: number; // Final amount to mediator
}

export class PaymentProcessingService {
  /**
   * Calculate all fees for a payment
   */
  calculateFees(grossAmount: number): FeeCalculation {
    // Stripe fee: 2.9% + $0.30
    const stripeFeeAmount = (grossAmount * STRIPE_CONFIG.STRIPE_FEE_PERCENT) + STRIPE_CONFIG.STRIPE_FEE_FIXED;
    const stripeNetAmount = grossAmount - stripeFeeAmount;
    
    // Platform fee: 10% of net after Stripe
    const platformFeeAmount = stripeNetAmount * STRIPE_CONFIG.PLATFORM_FEE_PERCENT;
    
    // GST on platform fee
    const platformFeeGst = platformFeeAmount * 0.10;
    
    // Amount to mediator
    const mediatorAmount = stripeNetAmount - platformFeeAmount - platformFeeGst;

    return {
      grossAmount: Math.round(grossAmount),
      stripeFeeAmount: Math.round(stripeFeeAmount),
      stripeNetAmount: Math.round(stripeNetAmount),
      platformFeeAmount: Math.round(platformFeeAmount),
      platformFeeGst: Math.round(platformFeeGst),
      mediatorAmount: Math.round(mediatorAmount),
    };
  }

  /**
   * Create deposit payment (platform keeps 100%)
   * Deposit is a flat $250 that goes entirely to the platform
   */
  async createDepositPayment(
    caseId: string,
    clientId: string,
    userId: string
  ) {
    // Get case and verify
    const caseRecord = await prisma.case.findUnique({
      where: { id: caseId },
      include: {
        mediator: true,
        parties: {
          where: { clientId },
        },
      },
    });

    if (!caseRecord) {
      throw new NotFoundError('Case not found');
    }

    if (caseRecord.parties.length === 0) {
      throw new ValidationError('Client is not a party to this case');
    }

    if (caseRecord.depositPaid) {
      throw new ValidationError('Deposit has already been paid');
    }

    // Create payment intent (no destination charge - platform keeps all)
    const paymentIntent = await stripe.paymentIntents.create({
      amount: STRIPE_CONFIG.DEPOSIT_AMOUNT,
      currency: STRIPE_CONFIG.CURRENCY,
      metadata: {
        type: 'DEPOSIT',
        caseId,
        clientId,
        mediatorId: caseRecord.mediatorId,
      },
      description: `Deposit for case ${caseRecord.caseNumber}`,
    });

    // Calculate fees (for tracking, though mediator gets $0)
    const fees = this.calculateFees(STRIPE_CONFIG.DEPOSIT_AMOUNT);

    // Create payment record
    const payment = await prisma.payment.create({
      data: {
        caseId,
        clientId,
        mediatorId: caseRecord.mediatorId,
        amount: new Decimal(STRIPE_CONFIG.DEPOSIT_AMOUNT / 100),
        stripeGrossAmount: new Decimal(fees.grossAmount / 100),
        stripeFeeAmount: new Decimal(fees.stripeFeeAmount / 100),
        stripeNetAmount: new Decimal(fees.stripeNetAmount / 100),
        platformFeeAmount: new Decimal(STRIPE_CONFIG.DEPOSIT_AMOUNT / 100), // Platform keeps all
        platformFeeGst: new Decimal(0),
        mediatorAmount: new Decimal(0), // Mediator gets nothing
        currency: STRIPE_CONFIG.CURRENCY,
        type: 'DEPOSIT',
        status: 'PENDING',
        stripePaymentIntentId: paymentIntent.id,
        description: 'Case deposit',
      },
    });

    await createAuditLog({
      userId,
      action: 'PAYMENT_CREATE',
      resource: 'payment',
      resourceId: payment.id,
      metadata: {
        type: 'DEPOSIT',
        amount: STRIPE_CONFIG.DEPOSIT_AMOUNT / 100,
        caseId,
      },
    });

    return { payment, paymentIntent };
  }

  /**
   * Create balance payment (destination charge to mediator)
   * Balance payments use destination charges with platform fee
   */
  async createBalancePayment(
    caseId: string,
    clientId: string,
    amount: number, // Amount in cents
    userId: string,
    description?: string
  ) {
    // Get case and verify mediator has Stripe Connect
    const caseRecord = await prisma.case.findUnique({
      where: { id: caseId },
      include: {
        mediator: true,
        parties: {
          where: { clientId },
        },
      },
    });

    if (!caseRecord) {
      throw new NotFoundError('Case not found');
    }

    if (caseRecord.parties.length === 0) {
      throw new ValidationError('Client is not a party to this case');
    }

    if (!caseRecord.mediator.stripeAccountId) {
      throw new ValidationError('Mediator has not set up payment account');
    }

    if (!caseRecord.mediator.stripeOnboarded) {
      throw new ValidationError('Mediator has not completed payment setup');
    }

    // Calculate fees
    const fees = this.calculateFees(amount);

    // Application fee is platform fee + GST
    const applicationFeeAmount = fees.platformFeeAmount + fees.platformFeeGst;

    // Create payment intent with destination charge
    const paymentIntent = await stripe.paymentIntents.create({
      amount: fees.grossAmount,
      currency: STRIPE_CONFIG.CURRENCY,
      application_fee_amount: applicationFeeAmount,
      transfer_data: {
        destination: caseRecord.mediator.stripeAccountId,
      },
      metadata: {
        type: 'HOURLY',
        caseId,
        clientId,
        mediatorId: caseRecord.mediatorId,
      },
      description: description || `Payment for case ${caseRecord.caseNumber}`,
    });

    // Create payment record
    const payment = await prisma.payment.create({
      data: {
        caseId,
        clientId,
        mediatorId: caseRecord.mediatorId,
        amount: new Decimal(amount / 100),
        stripeGrossAmount: new Decimal(fees.grossAmount / 100),
        stripeFeeAmount: new Decimal(fees.stripeFeeAmount / 100),
        stripeNetAmount: new Decimal(fees.stripeNetAmount / 100),
        platformFeeAmount: new Decimal(fees.platformFeeAmount / 100),
        platformFeeGst: new Decimal(fees.platformFeeGst / 100),
        mediatorAmount: new Decimal(fees.mediatorAmount / 100),
        currency: STRIPE_CONFIG.CURRENCY,
        type: 'HOURLY',
        status: 'PENDING',
        stripePaymentIntentId: paymentIntent.id,
        description: description || 'Balance payment',
      },
    });

    await createAuditLog({
      userId,
      action: 'PAYMENT_CREATE',
      resource: 'payment',
      resourceId: payment.id,
      metadata: {
        type: 'HOURLY',
        amount: amount / 100,
        caseId,
      },
    });

    return { payment, paymentIntent };
  }

  /**
   * Get payment by intent ID
   */
  async getPaymentByIntentId(intentId: string) {
    const payment = await prisma.payment.findUnique({
      where: { stripePaymentIntentId: intentId },
      include: {
        case: true,
      },
    });

    return payment;
  }

  /**
   * Update payment status
   */
  async updatePaymentStatus(
    paymentId: string,
    status: 'SUCCEEDED' | 'FAILED' | 'REFUNDED',
    metadata?: any
  ) {
    const payment = await prisma.payment.update({
      where: { id: paymentId },
      data: {
        status,
        paidAt: status === 'SUCCEEDED' ? new Date() : undefined,
        refundedAt: status === 'REFUNDED' ? new Date() : undefined,
      },
      include: {
        case: true,
      },
    });

    // If deposit succeeded, mark case as deposit paid
    if (payment.type === 'DEPOSIT' && status === 'SUCCEEDED') {
      await prisma.case.update({
        where: { id: payment.caseId },
        data: {
          depositPaid: true,
          depositAmount: payment.amount,
        },
      });
    }

    await createAuditLog({
      action: 'PAYMENT_CREATE',
      resource: 'payment',
      resourceId: paymentId,
      metadata: {
        status,
        type: payment.type,
        ...metadata,
      },
    });

    return payment;
  }

  /**
   * Get all payments for a case
   */
  async getCasePayments(caseId: string) {
    return prisma.payment.findMany({
      where: { caseId },
      include: {
        client: {
          select: {
            firstName: true,
            lastName: true,
          },
        },
      },
      orderBy: {
        createdAt: 'desc',
      },
    });
  }

  /**
   * Get all payments for a mediator
   */
  async getMediatorPayments(mediatorId: string) {
    return prisma.payment.findMany({
      where: { mediatorId },
      include: {
        case: {
          select: {
            caseNumber: true,
          },
        },
        client: {
          select: {
            firstName: true,
            lastName: true,
          },
        },
      },
      orderBy: {
        createdAt: 'desc',
      },
    });
  }

  /**
   * Get payment statistics for mediator
   */
  async getMediatorPaymentStats(mediatorId: string) {
    const payments = await prisma.payment.findMany({
      where: {
        mediatorId,
        status: 'SUCCEEDED',
      },
    });

    const totalEarnings = payments.reduce((sum, p) => {
      return sum + Number(p.mediatorAmount);
    }, 0);

    const totalProcessed = payments.reduce((sum, p) => {
      return sum + Number(p.stripeGrossAmount);
    }, 0);

    const totalPlatformFees = payments.reduce((sum, p) => {
      return sum + Number(p.platformFeeAmount) + Number(p.platformFeeGst);
    }, 0);

    return {
      totalPayments: payments.length,
      totalEarnings,
      totalProcessed,
      totalPlatformFees,
      averagePayment: payments.length > 0 ? totalEarnings / payments.length : 0,
    };
  }

  /**
   * Get all payments (admin)
   */
  async getAllPayments(filters?: {
    status?: string;
    type?: string;
    mediatorId?: string;
    startDate?: Date;
    endDate?: Date;
  }) {
    return prisma.payment.findMany({
      where: {
        ...(filters?.status && { status: filters.status as any }),
        ...(filters?.type && { type: filters.type as any }),
        ...(filters?.mediatorId && { mediatorId: filters.mediatorId }),
        ...(filters?.startDate && {
          createdAt: { gte: filters.startDate },
        }),
        ...(filters?.endDate && {
          createdAt: { lte: filters.endDate },
        }),
      },
      include: {
        case: {
          select: {
            caseNumber: true,
          },
        },
        client: {
          select: {
            firstName: true,
            lastName: true,
          },
        },
        mediator: {
          select: {
            firstName: true,
            lastName: true,
          },
        },
      },
      orderBy: {
        createdAt: 'desc',
      },
    });
  }

  /**
   * Get platform revenue statistics
   */
  async getPlatformStats() {
    const payments = await prisma.payment.findMany({
      where: { status: 'SUCCEEDED' },
    });

    const depositRevenue = payments
      .filter(p => p.type === 'DEPOSIT')
      .reduce((sum, p) => sum + Number(p.platformFeeAmount), 0);

    const balanceRevenue = payments
      .filter(p => p.type !== 'DEPOSIT')
      .reduce((sum, p) => sum + Number(p.platformFeeAmount) + Number(p.platformFeeGst), 0);

    const totalRevenue = depositRevenue + balanceRevenue;

    const totalProcessed = payments.reduce((sum, p) => {
      return sum + Number(p.stripeGrossAmount);
    }, 0);

    const totalToMediators = payments.reduce((sum, p) => {
      return sum + Number(p.mediatorAmount);
    }, 0);

    return {
      totalPayments: payments.length,
      totalProcessed,
      depositRevenue,
      balanceRevenue,
      totalRevenue,
      totalToMediators,
    };
  }
}
